#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2004, 2005, 2006, 2007, 2014 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
notifier wrapper for QT 4
"""

from __future__ import absolute_import

from typing import Dict, Optional  # noqa: F401

import PyQt5.Qt as qt

from . import Any, _DispathCB, _FileLike, _get_fd, _SocketCB, _TimerCB, dispatch, log  # noqa: F401

IO_READ = qt.QSocketNotifier.Read
IO_WRITE = qt.QSocketNotifier.Write
IO_EXCEPT = qt.QSocketNotifier.Exception

_qt_socketIDs = {
	IO_READ: {},
	IO_WRITE: {},
	IO_EXCEPT: {},
}  # type: Dict[int, Dict[_FileLike, Socket]] # map of sockets/condition/methods -> qt.QSocketNotifier

__min_timer = None
__exit = None  # type: Optional[int]


class NotifierErrorQtApplicationUnset(Exception):
	pass


class Socket(qt.QSocketNotifier):
	def __init__(self, socket, method, condition):
		# type: (_FileLike, _SocketCB, int) -> None
		qt.QSocketNotifier.__init__(self, _get_fd(socket), condition)
		self.method = method
		self.socket = socket
		self.activated.connect(self.notified)

	@qt.pyqtSlot(int)
	def notified(self, socket):
		# type: (_FileLike) -> None
		log.warn('QT: socket: %d event on socket %s' % (self.type(), str(socket)))
		if not self.method(self.socket):
			self.setEnabled(0)
			socket_remove(self.socket, self.type())


class Timer(qt.QTimer):
	def __init__(self, ms, method):
		# type: (int, _TimerCB) -> None
		if qt.QCoreApplication.instance() is None:
			# create a new Qt Application instance before calling timer_add, e.g.
			# app = qt.QCoreApplication([])
			raise NotifierErrorQtApplicationUnset()
		qt.QTimer.__init__(self, qt.QCoreApplication.instance())
		self.method = method
		self.timeout.connect(self.slotTick)
		self.start(ms)

	def slotTick(self):
		# type: () -> None
		try:
			if not self.method():
				self.stop()
				del self
		except BaseException as e:
			log.warn('TIMER FAILED: %s' % str(e))


_TimerID = Timer


def socket_add(socket, method, condition=IO_READ):
	# type: (_FileLike, _SocketCB, int) -> None
	"""The first argument specifies a socket, the second argument has to be a
	function that is called whenever there is data ready in the socket."""
	fd = _get_fd(socket)
	if any(fd == _get_fd(sock) for sock in _qt_socketIDs[condition]):
		log.warn('Socket %d already registered for condition %d' % (fd, condition))
		return
	_qt_socketIDs[condition][socket] = Socket(socket, method, condition)


def socket_remove(socket, condition=IO_READ):
	# type: (_FileLike, int) -> None
	"""Removes the given socket from scheduler."""
	if socket in _qt_socketIDs[condition]:
		_qt_socketIDs[condition][socket].setEnabled(0)
		del _qt_socketIDs[condition][socket]


def timer_add(interval, method):
	# type: (int, _TimerCB) -> _TimerID
	"""The first argument specifies an interval in milliseconds, the
	second argument a function. This function is called after
	interval milliseconds. If it returns True it's called again after
	interval milliseconds, otherwise it is removed from the
	scheduler."""
	return Timer(interval, method)


def timer_remove(id):
	# type: (_TimerID) -> None
	"""Removes _all_ function calls to the method given as argument from the
	scheduler."""
	if isinstance(id, Timer):
		id.stop()
		del id


def dispatcher_add(method, min_timeout=True):
	# type: (_DispathCB, bool) -> None
	global __min_timer
	__min_timer = dispatch.dispatcher_add(method, min_timeout)


def dispatcher_remove(method):
	# type: (_DispathCB) -> None
	global __min_timer
	__min_timer = dispatch.dispatcher_remove(method)


def loop():
	# type: () -> int
	"""Execute main loop forever."""
	while __exit is None:
		step()

	return __exit


def step(sleep=True, external=True):
	# type: (bool, bool) -> None
	if __min_timer and sleep:
		time = qt.QTime()
		time.start()
		qt.QCoreApplication.processEvents(qt.QEventLoop.AllEvents | qt.QEventLoop.WaitForMoreEvents, __min_timer)
		if time.elapsed() < __min_timer:
			qt.QThread.usleep(__min_timer - time.elapsed())
	else:
		qt.QCoreApplication.processEvents(qt.QEventLoop.AllEvents | qt.QEventLoop.WaitForMoreEvents)

	if external:
		dispatch.dispatcher_run()


def _exit(dummy, code=0):
	# type: (Any, int) -> None
	global __exit
	__exit = code


qt.QCoreApplication.exit = _exit
qt.QCoreApplication.quit = _exit
