#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2006, 2007, 2009, 2010 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
generic implementation of external dispatchers, integratable into
several notifiers.
"""

from typing import Callable, Dict, List, Optional  # noqa: F401

from . import _DispathCB

# required for dispatcher use
MIN_TIMER = 100

__dispatchers = {
	True: [],
	False: [],
}  # type: Dict[bool, List[_DispathCB]]


def dispatcher_add(method, min_timeout=True):
	# type: (_DispathCB, bool) -> Optional[int]
	"""The notifier supports external dispatcher functions that will be
	called within each scheduler step. This functionality may be useful
	for applications having an own event mechanism that needs to be
	triggered as often as possible. This method registers a new
	dispatcher function. To ensure that the notifier loop does not
	suspend to long in the sleep state during the poll a minimal timer
	MIN_TIMER is set to guarantee that the dispatcher functions are
	called at least every MIN_TIMER milliseconds."""
	__dispatchers[min_timeout].append(method)
	if __dispatchers[True]:
		return MIN_TIMER
	else:
		return None


def dispatcher_remove(method):
	# type: (_DispathCB) -> Optional[int]
	"""Removes an external dispatcher function from the list"""
	for val in (True, False):
		if method in __dispatchers[val]:
			__dispatchers[val].remove(method)
			break
	if __dispatchers[True]:
		return MIN_TIMER
	else:
		return None


def dispatcher_run():
	# type: () -> Optional[int]
	"""Invokes all registered dispatcher functions"""
	for val in (True, False):
		# there is no need to copy an empty dict
		if not __dispatchers[val]:
			continue
		for disp in __dispatchers[val][:]:
			if not disp():
				dispatcher_remove(disp)

	if __dispatchers[True]:
		return MIN_TIMER
	else:
		return None


def dispatcher_count():
	# type: () -> int
	return len(__dispatchers)
