#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2006 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
an example demonstrating the process handler class
"""

from __future__ import print_function

import sys
from typing import List, Sequence, Union, cast  # noqa: F401

import notifier
import notifier.popen

proc = None
output = []  # type: List[str]

lineno = 0


def stdout(pid, text):
	# type: (int, Union[Sequence[str], str]) -> None
	output.extend(text if isinstance(text, (list, tuple)) else [cast(str, text)])


def stderr(pid, text):
	# type: (int, Union[Sequence[str], str]) -> None
	for line in (text if isinstance(text, (list, tuple)) else [cast(str, text)]):
		print("(%d>2): %s" % (pid, line))


def died(pid, status):
	# type: (int, object) -> None
	global lineno
	print(">>> process %d died" % pid, status)

	if not output:
		print(">>> process %d produced NO output" % pid, status)
	elif lineno and len(output) != lineno:
		print('NUMBERS OF LINES DO NOT MATCH!', len(output), lineno)
		with open('ls_mismatch', 'w') as fd:
			fd.write('\n'.join(output))
		sys.exit(0)
	lineno = len(output)
	notifier.timer_add(100, run_ls)


def tick():
	# type: () -> bool
	return True


def runit():
	# type: () -> None
	global proc
	print('runit ...', end=' ')
	proc = notifier.popen.Process('/bin/sleep 5')
	proc = notifier.popen.Process('/bin/ls -ltr')
	proc.signal_connect('stdout', stdout)
	proc.signal_connect('stderr', stderr)
	proc.signal_connect('killed', died)
	proc.start()
	while True:
		if proc.is_alive():
			notifier.step()
		else:
			break


def run_ls():
	# type: () -> bool
	del output[:]
	proc = notifier.popen.Process('/bin/ls -latr --color=never /usr/lib')
	proc.signal_connect('stdout', stdout)
	proc.signal_connect('stderr', stderr)
	proc.signal_connect('killed', died)
	proc.start()

	return False


if __name__ == '__main__':
	notifier.init(notifier.GENERIC)

	# run a process and wait for its death
#	notifier.timer_add( 500, runit )

	# show we can still do things
	notifier.timer_add(100, tick)

	notifier.timer_add(500, run_ls)

	notifier.loop()
