/* This file contains code to calculate the libration in Lunar lat and long.
 * It is based entirely on code supplied to me 2 Oct 1996 by Stephen L. Moshier,
 * <moshier@world.std.com>, which implements a trigonometric expansion to
 * approximate the librations according to JPL DE403. The following doc file
 * was also supplied therewith:

This program calculates a trigonometric series adjusted for a best fit
to the lunar librations from the Jet Propulsion Laboratory's DE403
ephemeris.

lblon403.c and lblat403.c are series for the selenographic longitude
and latitude of the earth.  The longitude series matches DE403
with a maximum discrepancy of 2.6" (0.4" rms).  The latitude has a
maximum discrepancy of 1.9" (0.2" rms).  The two series are valid in
the Julian year interval 1600 to 2200.

In both Eckhardt and DE403 the moon's principal moment of inertia axes
form the lunar body coordinate system. The estimated geographic
locations of these axes vary as measurements and theories gradually
become more refined.  There is a significant difference between
Eckhardt and DE403 in the constant term of the longitude.  Almanac and
cartography coordinates drop the constant term (which is -64" for
DE403).  That practice gives librations relative to the mean
selenographic longitude of the earth, for which the discrepancy from
one theory to the next is small.

Steve Moshier
October, 1996

 * Any errors in adapting this code to xephem are strictly my own.
 * Elwood Downey.
*/

#include <stdio.h>
#include <math.h>

#include "astro.h"

#define CHAR short


/* plantbl.h */

struct plantbl {
  /*  char max_harmonic[9]; */
  char max_harmonic[14];
  char max_power_of_t;
  CHAR *arg_tbl;
  int *lon_tbl;
  int *lat_tbl;
  int *rad_tbl;
  double distance;
  double timescale;
  double trunclvl;
};


/* libra403.c */

static double JD2000 = 2451545.0;
/* Conversion factors between degrees and radians */
static double STR = 4.8481368110953599359e-6;	/* radians per arc second */

static double ss[14][24];
static double cc[14][24];

/* Reduce arc seconds modulo 360 degrees,
   answer in arc seconds.  */
static double
mods3600 (double x)
{
  double y;
  y = x - 1.296e6 * floor (x / 1.296e6);
  return y;
}

/* Prepare lookup table of sin and cos ( i * L_k )
   for required multiple angles.
   K is the array offset corresponding to the planet.
   ARG is in radians.
   n is the highest harmonic to compute.  */
static int
sscc (int k, double arg, int n)
{
  double cu, su, cv, sv, s;
  int i;

  if (n <= 0)
    return 0;

  su = sin (arg);
  cu = cos (arg);
  ss[k][0] = su;		/* sin(L) */
  cc[k][0] = cu;		/* cos(L) */
  sv = 2.0 * su * cu;
  cv = cu * cu - su * su;
  ss[k][1] = sv;		/* sin(2L) */
  cc[k][1] = cv;
  for (i = 2; i < n; i++)
    {
      s = su * cv + cu * sv;
      cv = cu * cv - su * sv;
      sv = s;
      ss[k][i] = sv;		/* sin( i+1 L ) */
      cc[k][i] = cv;
    }
  return (0);
}


/* Mean elements.
   Copied from cmoon.c, DE404 version.  */

static double Jlast = -1.0e38;
static double T;

static int
dargs (double J, struct plantbl *plan)
{
  double T2, w;

  if (J == Jlast)
    return 0;

  T = (J - JD2000) / 36525.0;
  T2 = T * T;

  /* Mean anomaly of sun = l' (J. Laskar) */
  w = mods3600 (129596581.038354 * T + 1287104.76154);
  w += ((((((((
		1.62e-20 * T
		- 1.0390e-17) * T
	      - 3.83508e-15) * T
	     + 4.237343e-13) * T
	    + 8.8555011e-11) * T
	   - 4.77258489e-8) * T
	  - 1.1297037031e-5) * T
	 + 1.4732069041e-4) * T
	- 0.552891801772) * T2;
  sscc (10, STR * w, plan->max_harmonic[10]);

  /* Mean distance of moon from its ascending node = F */
  w = mods3600 ((1739527263.0983 - 2.079419901760e-01) * T + 335779.55755);
  w = w + ((-9.646018347184e-06 * T2	/* F, t^4 */
	    - 1.138215912580e-03) * T	/* F, t^3 */
	   - 1.312045233711e+01) * T;	/* F, t^2 */
  sscc (11, STR * w, plan->max_harmonic[11]);

  /* Mean anomaly of moon = l */
  w = mods3600 ((1717915923.4728 - 2.035946368532e-01) * T + 485868.28096);
  w = w + ((-3.421689790404e-04 * T2	/* l, t^4 */
	    + 4.768357585780e-02) * T	/* l, t^3 */
	   + 3.146734198839e+01) * T;	/* l, t^2 */
  sscc (9, STR * w, plan->max_harmonic[9]);

  /* Mean elongation of moon = D */
  w = mods3600 ((1602961601.4603 + 3.962893294503e-01) * T + 1072260.73512);
  w = w + ((-2.905334122698e-04 * T2	/* D, t^4 */
	    - 5.834100476561e-03) * T	/* D, t^3 */
	   - 6.847070905410e+00) * T;	/* D, t^2 */
  sscc (12, STR * w, plan->max_harmonic[12]);

  /* Mean longitude of moon, re mean ecliptic and equinox of date */
  w = mods3600 ((1732564372.83264 - 6.784914260953e-01) * T + 785939.95571);
  w = w + ((-8.466472828815e-05 * T2	/* L, t^4 */
	    + 5.722859298199e-03) * T	/* L, t^3 */
	   - 5.663161722088e+00) * T;	/* L, t^2 */
  sscc (13, STR * w, plan->max_harmonic[13]);

  /* Mean longitudes of planets (Laskar, Bretagnon) */

  /* Venus.  */
  w = mods3600 (210664136.4335482 * T + 655127.283046);
  w += ((((((((
		-9.36e-023 * T
		- 1.95e-20) * T
	      + 6.097e-18) * T
	     + 4.43201e-15) * T
	    + 2.509418e-13) * T
	   - 3.0622898e-10) * T
	  - 2.26602516e-9) * T
	 - 1.4244812531e-5) * T
	+ 0.005871373088) * T2;
  sscc (1, STR * w, plan->max_harmonic[1]);

  /* Earth.  */
  w = mods3600 (129597742.26669231 * T + 361679.214649);
  w += ((((((((-1.16e-22 * T
	       + 2.976e-19) * T
	      + 2.8460e-17) * T
	     - 1.08402e-14) * T
	    - 1.226182e-12) * T
	   + 1.7228268e-10) * T
	  + 1.515912254e-7) * T
	 + 8.863982531e-6) * T
	- 2.0199859001e-2) * T2;
  sscc (2, STR * w, plan->max_harmonic[2]);

  /* Mars.  */
  w = mods3600 (68905077.59284 * T + 1279559.78866);
  w += (-1.043e-5 * T + 9.38012e-3) * T2;
  sscc (3, STR * w, plan->max_harmonic[3]);

  /* Jupiter.  */
  w = mods3600 (10925660.428608 * T + 123665.342120);
  w += (1.543273e-5 * T - 3.06037836351e-1) * T2;
  sscc (4, STR * w, plan->max_harmonic[4]);

  /* Saturn.  */
  w = mods3600 (4399609.65932 * T + 180278.89694);
  w += ((4.475946e-8 * T - 6.874806E-5) * T + 7.56161437443E-1) * T2;
  sscc (5, STR * w, plan->max_harmonic[5]);
  return 0;
}



/* Evaluate series PLAN at Julian date J.
   Result in arcseconds, usually.  */

static double
gplan (double J, struct plantbl *plan)
{
  double su, cu, sv, cv;
  double t, sl;
  int j, k, m, k1, ip, np, nt;
  CHAR *p;
  int *pl;

  dargs (J, plan);
  /* Point to start of table of arguments. */
  p = plan->arg_tbl;
  /* Point to tabulated cosine and sine amplitudes.  */
  pl = plan->lon_tbl;
  sl = 0.0;

  for (;;)
    {
      /* Find sine and cosine of argument for this term in the series.
	 The argument has the form J_1 L_1 + J_2 L_2 + ...
	 where J_i are integers and L_i are mean elements.  */

      /* Number of periodic arguments. */
      np = *p++;
      if (np < 0)
	break;
      if (np == 0)
	{
	  /* If no periodic arguments, it is a polynomial term.
	     Evaluate A_n T^n + A_n-1 T^n-1 + ... + A_0.  */
	  nt = *p++;
	  cu = *pl++;
	  for (ip = 0; ip < nt; ip++)
	    cu = cu * T + *pl++;
	  sl += cu;
	  continue;
	}
      k1 = 0;
      cv = 0.0;
      sv = 0.0;
      for (ip = 0; ip < np; ip++)
	{
	  /* What harmonic.  */
	  j = *p++;
	  /* Which planet.  */
	  m = *p++ - 1;
	  if (j)
	    {
	      k = j;
	      if (j < 0)
		k = -k;
	      k -= 1;
	      /* sin(k*angle) for planet m.  */
	      su = ss[m][k];
	      if (j < 0)
		su = -su;
	      /* cos(k*angle) for planet m.  */
	      cu = cc[m][k];
	      if (k1 == 0)
		{
		  /* Set sine and cosine of first angle. */
		  sv = su;
		  cv = cu;
		  k1 = 1;
		}
	      else
		{
		  /* Combine angles by trigonometry.  */
		  t = su * cv + cu * sv;
		  cv = cu * cv - su * sv;
		  sv = t;
		}
	    }
	}
      /* Now cv = cos(arg), sv = sin(arg).
	 Evaluate
	 cu = (C_n T^n + C_n-1 T^n-1 + ... + C_0) cos(arg)
	 su = (S_n T^n + S_n-1 T^n-1 + ... + S_0) sin(arg).  */

      /* Highest power of T.  */
      nt = *p++;
      /* Coefficients C_i, S_i.  */
      cu = *pl++;
      su = *pl++;
      for (ip = 0; ip < nt; ip++)
	{
	  cu = cu * T + *pl++;
	  su = su * T + *pl++;
	}
      sl += cu * cv + su * sv;
    }
  return (plan->trunclvl * sl);
}


/* lblat403.c */

static int lattabr[] = {-1};
static int lattabb[] = {-1};
static int lattabl[] = {
    785710,

     -1958,       780,

      -178,      2850,      2243,      5927,      2714,     -2911,

      -174,      5888,    -16705,     10828,    -15113,    -25049,

      -172,      1976,

        37,    -13560,

       -40,      -911,

      -300,       691,

      3779,       722,

        68,     -1048,

     -3092,        -4,      3011,       -31,       250, -14293664,

      -149,      -228,

        -5,      -842,

        62,     57967,

        -7,    -11207,

        15,    -66055,

       -79,    -71560,

         8,      2998,

     -1062,       -46,

        -2,       711,

        -5,        -2,

       367,        -5,      -353,         1,      -117,   2324027,

         0,      -176,

        19,        -4,

        -3,     -1229,

         7,      4327,

        -2,       112,

        -1,      -540,

        73,    102982,

         0,      -137,

         2,       239,

         0,      -216,

         2,      3707,

        -1,      -100,

        -2,       124,

         0,       408,

         1,       561,

         0,       594,

         0,     11866,

        -2,      -229,

        11,      6768,

         2,         1,

        -2,      -788,       -24,    314732,

        -5,     -3238,

        -2,     -1263,

        -1,        94,

        -2,       206,

      -283,   6489836,

        63,      2562,

        -6,      1220,

         0,       135,

       795,      -338,

      -406,      -101,

       757,      -383,

       -73,    -31018,

        -5,     -3010,

         6,      5821,

       167,         7,

         2,    122623,

        41,    138924,

         2,       253,

         0,      -198,

       135,        30,

         2,       934,

        62,       -14,

      -531,   -297537,

        -3,      -205,

         1,      -393,

        -4,      -156,

      -467,        90,

         9,      -346,

       -55,      1368,

      4414,      2094,

       480,       510,

        35,       728,

      6832,      6429,

     -2099,     -3703,

        -9,       579,

       239,      1738,

        22,       968,     60260,     -8929,

        26,       -64,

         0,         0,

       -47,      -351,      2248,     -1175,      3976,      7150,

     15191,         1,    -15535,      -143,     -2308,-239959618,

       -45,       355,      2252,      1189,      4009,     -7131,

      -158,       499,

       776,       -34,

      -389,      4363,

        30,        28,

         0,      -593,

         1,        -1,

        58,       -66,

         4,      1552,

        -1,       244,

       288,        59,

      -276,        62,

         8,      3309,

         2,      1249,

      -156,        -3,

         0,     -7903,

       -47,    154458,

      -710,       160,

         0,       288,

       328,    219514,

        -1,       113,

        -1,      1515,

       -20,     28493,

        13,      9011,

        -1,      1110,

        -1,      -103,

         0,       312,

         1,       360,

         6,      6139,

        -1,       142,

        -1,      -709,

        -1,      -242,

        67,     67200,

         5,      4148,

         1,       137,

       -15,    -17969,

        -2,     -3372,

        -2,     -1739,

       154,    116200,

        -1,      -640,

      -269,   -336274,

         0,     -1557,

        -1,      -353,

       993,        39,

        -1,      -294,

      -646,      -135,

       750,        -3,      -661,         8,      -163,   2591179,

        -1,      -258,

         0,       381,

         6,      8272,

       -44,    -92044,

       -29,     70397,

       -25,    -17892,

         1,        67,

       -44,    -17118,

      2743,       349,

        -4,       231,

      1501,      3099,

       -33,        21,

     -1130,        97,      1263,       164,      -226, -13139965,

       839,      -202,

        -1,       660,

        12,    -10482,

         1,        17,

        -2,     -1480,

       -45,    -26084,

         2,       259,

         0,       258,

        -1,     -1560,

        32,     73046,

        -1,      -370,

        -1,       -77,

        46,     20687,

         1,       742,

         1,      2399,

         2,       741,

         0,        52,

         0,       112,

         0,       297,

         2,       825,

         0,       161,

         1,      4228,

         0,       212,

        -1,      -105,

         6,      2782,

         9,     37199,

         3,      5045,

       105,        -1,

        -1,      -179,

        75,     31274,

        -3,     -1321,

        -1,       363,

         0,      1137,

        58,    104090,

        -2,     -2695,

        -1,      -392,

       -35,    -21763,

        -2,      -205,

       567,        38,

        -7,       117,

      -569,        -9,       321,  -1522061,

        -4,       276,

        -2,      -590,

         2,       632,

       -14,    -10601,

         4,      8703,

       -10,     16870,

        -4,      -995,

       597,        -6,      -386,   -805403,

      1025,      -474,

        -2,       201,

       -95,        16,

      -127,    135570,

         0,      -173,

        10,      8565,

        -3,      1812,

        -1,      -108,

      -107,      -649,

         0,       151,

         1,       950,

         1,       312,

         1,       403,

         0,      -576,

         7,      4418,

        -3,     -4769,

        -1,      -769,

        63,     38890,

         0,      -101,

         0,       169,

        -3,    -14837,

         1,       372,

         0,      -674,

        -3,     -1488,

       -36,   -196300,

         0,      -176,

       -10,     -4811,

        -2,      -991,

         1,       831,

         1,      3138,

      -103,    -51819,

         0,      -133,

        -4,     22171,

         2,       820,

         1,       149,

      2227,      1277,

      -502,     -3049,

         1,       122,

         0,       370,

         0,       166,

         2,       772,

         2,      2057,

        -1,      -486,

        13,    -15447,

        -2,     -1619,

         0,       335,

         0,      -206,

         0,      -104,

       -19,    -19726,

         0,       414,

        -8,     -3425,

       -19,        -1,

         1,       546,

         0,      -392,

         0,     -2754,

         0,      -158,

        -3,     -1790,

        -1,      -230,

         0,      -190,

         0,      -366,

         0,      -154,

};

static CHAR latargs[] = {
  0,  0,
  3,  1,  2,  9,  3,-20,  4,  0,
  4,  1, 13, -1, 10,  3,  2, -4,  3,  2,
  4,  1, 13, -1, 10, -5,  2,  9,  3,  2,
  3,  9, 10, -9, 14,  1,  3,  0,
  3,  1, 11, -1, 12,  1, 13,  0,
  4,  1, 10, -2, 11,  1, 12, -2, 13,  0,
  4,  2, 11, -1, 14, 18,  2,-18,  3,  0,
  3,  1, 13, -1, 10,  1,  3,  0,
  2,  1, 10, -1, 14,  0,
  2,  1, 10, -1, 12,  2,
  4,  1, 14,-18,  2, 16,  3,  1,  5,  0,
  3,  2, 10, -1, 12, -1, 13,  0,
  3,  1, 10,  1, 11, -1, 12,  0,
  4,  1, 10, -1, 11,  1, 12, -2, 13,  0,
  2,  1, 12, -1, 13,  0,
  3,  1, 10, -1, 11, -1, 12,  0,
  3,  3, 10, -1, 12, -2, 13,  0,
  2,  2, 10, -2, 13,  0,
  3,  1, 10,  2, 11, -1, 12,  0,
  3,  1, 10, -2, 13,  1,  3,  0,
  3,  1, 10,  1, 12, -2, 13,  2,
  3,  1, 11,  1, 12, -1, 13,  0,
  2,  2, 12, -2, 13,  0,
  3,  1, 10, -2, 11, -1, 12,  0,
  3,  1, 10, -3, 12,  2, 13,  0,
  4,  3, 10,  1, 11, -1, 12, -2, 13,  0,
  3,  2, 10,  1, 12, -3, 13,  0,
  4,  1, 10,  1, 11,  1, 12, -2, 13,  0,
  2,  3, 12, -3, 13,  0,
  4,  1, 10, -1, 11, -3, 12,  2, 13,  0,
  3,  3, 10,  1, 12, -4, 13,  0,
  4,  1, 10,  2, 11,  1, 12, -2, 13,  0,
  3,  1, 10,  3, 12, -4, 13,  0,
  4,  1, 10,  3, 11,  1, 12, -2, 13,  0,
  3,  3, 11,  1, 12, -2, 13,  0,
  4,  2, 10,  1, 11,  1, 12, -4, 13,  0,
  2,  3, 12, -4, 13,  0,
  3,  2, 11,  1, 12, -2, 13,  0,
  4,  1, 10,  1, 11,  1, 12, -3, 13,  0,
  3,  2, 10,  1, 12, -4, 13,  0,
  4,  1, 14, -3,  2,-16,  3, -4,  4,  0,
  3,  1, 11,  1, 12, -2, 13,  1,
  3,  1, 10,  1, 12, -3, 13,  0,
  4,  2, 10,  1, 11, -1, 12, -2, 13,  0,
  3,  2, 10, -3, 12,  2, 13,  0,
  4,  1, 10, -1, 11, -1, 12,  1, 13,  0,
  2,  1, 12, -2, 13,  0,
  2,  2, 13, -1, 14,  0,
  2,  2, 11, -1, 12,  0,
  4,  1, 10, -1, 11,  1, 12, -3, 13,  0,
  3,  2, 10, -3, 14,  4,  4,  0,
  2,  1, 10, -2, 13,  0,
  3,  1, 13, -3,  3,  4,  4,  0,
  3,  2, 10, -1, 12, -2, 13,  0,
  3,  2, 10, -1, 11, -1, 12,  0,
  3,  1, 10, -1, 12,  1, 13,  0,
  2,  1, 10, -1, 11,  0,
  3,  1, 11, -1, 12,  2, 13,  0,
  2,  1, 11, -1, 12,  0,
  3,  1, 10, -1, 12, -1, 13,  0,
  4,  2, 13, -1, 12,  2,  2, -2,  3,  0,
  3,  1, 13,  2,  2, -3,  3,  0,
  4,  2, 10, -1, 11, -1, 12, -2, 13,  0,
  3,  1, 13, -1,  2,  2,  3,  0,
  2,  2, 10, -1, 12,  0,
  4,  2, 13, -1, 12,  2,  3, -2,  5,  0,
  4,  2, 13, -1, 12,  3,  2, -3,  3,  0,
  4,  2, 10, -2, 11,  1, 12, -2, 13,  0,
  3,  1, 13,  1,  3, -2,  5,  0,
  4,  1, 10,  1, 11, -1, 12,  1, 13,  0,
  3,  1, 13, -5,  2,  9,  3,  0,
  3,  1, 14,  2,  3, -4,  4,  0,
  3,  3, 10,  4, 12, -6, 14,  0,
  3,  1, 13,  3,  2, -4,  3,  0,
  1,  1, 10,  0,
  3,  1, 13, -3, 12,  9,  2,  0,
  4,  1, 10, -1, 11,  1, 12, -1, 13,  0,
  3,  2, 11, -1, 12,  2, 13,  0,
  1,  1, 14,  1,
  4,  1, 12,  1, 10, -1, 14,  1,  5,  0,
  4,  9, 13, -9, 10,  7,  2, 10,  3,  0,
  3,  1, 12, -8,  2, 13,  3,  2,
  1,  1, 12,  2,
  3,  1, 12,  8,  2,-13,  3,  2,
  2,  1, 13,  2,  4,  0,
  4,  2, 13, -1, 12,  5,  2, -6,  3,  0,
  3,  1, 13, -2, 12,  1,  3,  0,
  3,  1, 13, -3,  2,  6,  3,  0,
  4,  1, 10, -1, 11, -1, 12, -1, 13,  0,
  2,  1, 14,  2,  5,  0,
  2,  1, 10, -2, 12,  0,
  2,  2, 10, -3, 12,  0,
  4,  2, 13, -1, 12,  4,  2, -4,  3,  0,
  3,  1, 14,  4,  2, -6,  3,  0,
  2,  1, 13,  1,  2,  0,
  3,  2, 10,  1, 11, -1, 12,  0,
  4,  2, 10, -1, 11,  1, 12, -2, 13,  0,
  2,  1, 10,  1, 11,  0,
  3,  1, 10,  1, 12, -1, 13,  0,
  2,  1, 11,  1, 12,  0,
  3,  3, 12, -2, 13, -1,  3,  0,
  3,  4, 10, -1, 12, -2, 13,  0,
  3,  2, 10,  1, 12, -2, 13,  0,
  4,  1, 10,  1, 11,  1, 12, -1, 13,  0,
  2,  2, 11,  1, 12,  0,
  2,  3, 12, -2, 13,  0,
  4,  2, 10,  1, 11,  1, 12, -2, 13,  0,
  3,  1, 11,  3, 12, -2, 13,  0,
  3,  4, 10,  1, 12, -4, 13,  0,
  4,  2, 10,  2, 11,  1, 12, -2, 13,  0,
  4,  1, 10,  2, 11,  1, 12, -4, 13,  0,
  4,  1, 10,  1, 11,  1, 12, -4, 13,  0,
  4,  1, 10,  3, 11, -1, 12, -2, 13,  0,
  4,  1, 10, -2, 11, -1, 12,  2, 13,  0,
  3,  1, 11,  1, 12, -3, 13,  0,
  3,  1, 10,  1, 12, -4, 13,  0,
  4,  1, 10,  2, 11, -1, 12, -2, 13,  0,
  3,  3, 10, -1, 12, -4, 13,  0,
  4,  1, 10, -1, 11, -1, 12,  2, 13,  0,
  2,  1, 12, -3, 13,  0,
  4,  1, 10, -1, 11,  1, 12, -4, 13,  0,
  4,  1, 10,  1, 11, -1, 12, -2, 13,  0,
  3,  2, 10, -1, 12, -3, 13,  0,
  3,  1, 10, -1, 12,  2, 13,  0,
  3,  1, 10, -2, 11,  1, 12,  0,
  3,  1, 11, -1, 12,  3, 13,  0,
  1,  2, 13,  0,
  3,  1, 11, -1, 12, -1, 13,  0,
  3,  1, 10, -3, 13, -1,  3,  0,
  3,  1, 10, -1, 12, -2, 13,  2,
  3,  3, 10, -1, 11, -1, 12,  0,
  3,  2, 10, -1, 12,  1, 13,  0,
  4,  1, 10,  1, 11, -1, 12,  2, 13,  0,
  3,  1, 10, -1, 11,  1, 12,  0,
  2,  1, 12,  1, 13,  0,
  4,  1, 10, -1, 11, -1, 12, -2, 13,  0,
  3,  1, 10,  1, 11, -3, 12,  0,
  2,  3, 10, -1, 12,  0,
  1,  2, 10,  0,
  4,  2, 10, -1, 11,  1, 12, -1, 13,  0,
  3,  1, 13,  1, 10,  1, 11,  0,
  3,  2, 13,  3,  3, -2, 11,  0,
  2,  1, 10,  1, 12,  2,
  3,  1, 10, -3, 14,  2,  5,  0,
  3,  3, 10, -1, 14,  4,  5,  0,
  3,  1, 11,  1, 12,  1, 13,  0,
  1,  2, 12,  0,
  4,  1, 10, -2, 11, -1, 12, -2, 13,  0,
  2,  1, 10, -3, 12,  0,
  3,  3, 10,  1, 11, -1, 12,  0,
  4,  3, 10, -1, 11,  1, 12, -2, 13,  0,
  3,  2, 10,  1, 12, -1, 13,  0,
  3,  1, 10,  1, 11,  1, 12,  0,
  2,  3, 12, -1, 13,  0,
  3,  1, 10, -1, 11, -3, 12,  0,
  3,  3, 10,  1, 12, -2, 13,  0,
  3,  1, 10,  2, 11,  1, 12,  0,
  3,  1, 10,  3, 12, -2, 13,  0,
  4,  3, 10,  1, 11,  1, 12, -2, 13,  0,
  4,  1, 10,  1, 11,  3, 12, -2, 13,  0,
  4,  2, 10,  1, 11,  1, 12, -6, 13,  0,
  3,  2, 11,  1, 12, -4, 13,  0,
  3,  2, 10,  1, 12, -6, 13,  0,
  4,  2, 10,  2, 11, -1, 12, -4, 13,  0,
  3,  1, 11,  1, 12, -4, 13,  0,
  3,  3, 11, -1, 12, -2, 13,  0,
  3,  1, 10,  1, 12, -5, 13,  0,
  4,  2, 10,  1, 11, -1, 12, -4, 13,  0,
  2,  1, 12, -4, 13,  0,
  3,  2, 11, -1, 12, -2, 13,  0,
  2,  1, 10, -4, 13,  0,
  4,  1, 10,  1, 11, -1, 12, -3, 13,  0,
  3,  2, 10, -1, 12, -4, 13,  0,
  4,  2, 10, -1, 11, -1, 12,  2, 13,  0,
  3,  1, 10, -1, 12,  3, 13,  0,
  3,  1, 11, -1, 12,  4, 13,  0,
  3,  1, 11, -1, 12, -2, 13,  0,
  3,  1, 10, -1, 12, -3, 13,  0,
  4,  2, 10, -1, 11, -1, 12, -4, 13,  0,
  3,  2, 10, -1, 12,  2, 13,  0,
  3,  2, 10, -2, 11,  1, 12,  0,
  2,  1, 10,  2, 13,  0,
  4,  1, 10, -1, 11,  1, 12,  1, 13,  0,
  2,  1, 12,  2, 13,  1,
  4,  1, 10, -1, 11, -1, 12, -3, 13,  0,
  3,  2, 10, -3, 12, -2, 13,  0,
  4,  2, 10,  1, 11, -1, 12,  2, 13,  0,
  3,  2, 10, -1, 11,  1, 12,  0,
  3,  1, 10,  1, 12,  1, 13,  0,
  3,  1, 11,  1, 12,  2, 13,  0,
  2,  4, 10, -1, 12,  0,
  2,  2, 10,  1, 12,  1,
  4,  1, 13,  1, 12, 10,  2, -2,  3,  0,
  3,  2, 11,  1, 12,  2, 13,  0,
  2,  2, 12,  1, 14,  0,
  1,  3, 12,  0,
  3,  3, 10,  1, 12, -1, 13,  0,
  3,  2, 10,  1, 11,  1, 12,  0,
  3,  4, 10,  1, 12, -2, 13,  0,
  3,  2, 10,  3, 12, -2, 13,  0,
  3,  4, 10, -9,  3, -1,  4,  0,
  4,  1, 10,  1, 11,  1, 12, -6, 13,  0,
  3,  1, 10,  1, 12, -6, 13,  0,
  4,  1, 10,  2, 11, -1, 12, -4, 13,  0,
  3,  3, 10, -1, 12, -6, 13,  0,
  4,  1, 10, -1, 11, -1, 12,  4, 13,  0,
  4,  1, 10,  1, 11, -1, 12, -4, 13,  0,
  3,  1, 10, -1, 12,  4, 13,  0,
  4,  1, 10, -2, 11,  1, 12,  2, 13,  0,
  3,  1, 10, -1, 12, -4, 13,  0,
  4,  3, 10, -1, 11, -1, 12,  2, 13,  0,
  4,  1, 10,  1, 11, -1, 12,  4, 13,  0,
  4,  1, 10, -1, 11,  1, 12,  2, 13,  0,
  2,  1, 12,  3, 13,  0,
  4,  1, 10, -1, 11, -1, 12, -4, 13,  0,
  3,  3, 10, -1, 12,  2, 13,  0,
  3,  1, 10,  1, 12,  2, 13,  0,
  3,  1, 11,  1, 12,  3, 13,  0,
  3,  1, 10, -3, 12, -2, 13,  0,
  3,  3, 10, -1, 11,  1, 12,  0,
  3,  2, 10,  1, 12,  1, 13,  0,
  4,  1, 10,  1, 11,  1, 12,  2, 13,  0,
  2,  3, 10,  1, 12,  0,
  4,  2, 10,  1, 11,  1, 12,  1, 13,  0,
  2,  1, 10,  3, 12,  0,
  3,  3, 10,  1, 11,  1, 12,  0,
  3,  5, 10,  1, 12, -2, 13,  0,
  3,  4, 14, -4,  2,  8,  3,  0,
  3,  5, 10, -9,  3, -1,  4,  0,
  4,  2, 10,  1, 11, -1, 12, -6, 13,  0,
  2,  1, 12, -6, 13,  0,
  3,  2, 11, -1, 12, -4, 13,  0,
  3,  2, 10, -1, 12, -6, 13,  0,
  3,  1, 11, -1, 12, -4, 13,  0,
  3,  2, 10, -1, 12,  4, 13,  0,
  2,  1, 12,  4, 13,  0,
  4,  2, 10, -1, 11,  1, 12,  2, 13,  0,
  3,  1, 11,  1, 12,  4, 13,  0,
  3,  1, 11, -3, 12, -2, 13,  0,
  3,  4, 10, -1, 12,  2, 13,  0,
  3,  2, 10,  1, 12,  2, 13,  0,
  4,  2, 10,  1, 11,  1, 12,  2, 13,  0,
  2,  4, 10,  1, 12,  0,
  3,  7, 12, -2, 10, -4,  5,  0,
  3,  1, 10, -1, 12, -6, 13,  0,
  4,  1, 10, -1, 11,  1, 12,  4, 13,  0,
  3,  1, 10,  1, 12,  4, 13,  0,
  4,  3, 10, -1, 11,  1, 12,  2, 13,  0,
  3,  3, 10,  1, 12,  2, 13,  0,
  2,  5, 10,  1, 12,  0,
  2,  1, 12,  6, 13,  0,
  3,  2, 10,  1, 12,  4, 13,  0,
  3,  4, 10,  1, 12,  2, 13,  0,
 -1
};

/* Total terms = 254, small = 254 */
static struct plantbl liblat = {
  /*  {  0, 18, 18, 20,  4,  0,  0,  0,  0,  9,  3,  7,  9,  9,}, */
  /* Use max of liblon, liblat.  */
  {  0, 18, 18, 20,  4,  2,  0,  0,  0,  9, 16,  7,  9,  9,},
 2,
 latargs,
 lattabl,
 lattabb,
 lattabr,
 3.850000e+05,
 36525.0,
 1.0e-4,
};

/* lblon403.c */
static int lontabr[] = {-1};
static int lontabb[] = {-1};
static int lontabl[] = {
   -640001,

       -19,         3,

        -1,         0,

        -3,        17,

         2,        -9,

         2,         3,

       141,      1816,

         2,       -76,

        78,       -81,

         3,         1,

        50,       -13,

       -12,        18,

        -6,        74,

        19,        10,

       -19,       701,

       -26,        37,

        36,      -122,

       131,       -71,

       -40,         1,

     61463,     12853,

         5,       -29,

       354,       201,

        94,        -5,

        31,        25,

       -73,       -22,

       673,      1435,

      3844,        44,

       -44,        -8,

       195,       -16,

      -827,   -171768,

      7051,     -4116,

      1036,       616,     -2034,      6300,    -13786,     -3808,

      -118,      -536,     -1246,       771,      2555,      3214,

       203,        26,

      2975,      -715,

      -743,      1286,

      -232,       -50,

         4,       234,

      -139,       475,

         0,       -98,

        -3,    -11848,

         0,       118,

      -202,       146,

    -33673,      7601,

      -150,        88,

     -1720,      1002,

      -293,       481,

     -2078,         1,

       199,    220655,

       137,      7459,

      -127,        28,

       -10,      2259,

         4,       450,

       -26,        30,       -76,     19043,       215,  -7577830,

       -65,       -45,

        -5,    -14189,

        -6,      -496,

         9,      3265,

         0,      -131,

        48,       -90,

        -9,      -155,

        -2,        21,

       151,        19,

      -101,      -516,

      -827,         3,       796,         2,       134,  -2215850,

       -11,     -1908,

         0,       451,

        -2,      1399,

         3,         7,

      -163,         5,

     -1582,        61,

       -22,    -77215,

       167,   -471084,

       -85,       -28,

         1,      -473,

        -1,       444,

         4,        89,

         6,       -88,

      -157,    -87813,

         3,       274,

        -3,     -1036,

         7,    -17005,

        -1,      -332,

         0,        50,

        -1,      -228,

        -5,     -2986,

        -1,      -334,

         1,      -452,

         0,         4,

         0,      -247,

        -3,     -2493,

        -4,     -1000,

         0,      -160,

         0,     -3163,

       -75,    -74207,

        13,       829,

        32,       481,

       -33,    -11859,

        -2,     -3634,

         7,     -1611,

         3,       520,

        -1,      2534,

         5,      -446,

      -471,         8,       399,      5160,       201,  -2057189,

        83,     -2106,

        11,     12240,

         3,      -270,

       248,      1112,

       -63,    -61809,

      -183,       594,

        27,     25844,

        -3,      5594,

         8,       571,

       -23,     -1855,

       477,      -466,

         0,     -1257,

       247,      1550,

    -10178,      -175,      9570,      -423,      2610, -45917681,

       -51,       937,

     -1001,       168,

      -168,       882,

      1260,     -2894,      4306,     -1515,

         7,       402,

         0,        36,

      -280,      -373,

       -19,       840,

     -1227,      -138,

      3486,   1478247,

      1649,      -644,

       840,       377,

        47,     -1464,

       167,  -1251123,

      -194,      -713,       133,    284429,

         0,      -191,

        -3,      1534,

      -700,       918,

        -7,      -633,

      -221,      -559,

         1,        99,

       -17,     -3513,

       -30,        14,

      -602,     -5006,

       908,      4178,

       542,      -379,

      1568,       782,       528,      2761,

    -38994,    -25804,

   1696398,   -324306,

     47292,     71061,    457883,     52470,    159777, 224906030,

    -13068,    -18508,

      -104,      3754,

       541,       614,      -290,      1412,

       -48,       162,

        42,       -50,

      -609,     -4969,

        44,    -11435,

      -473,      -412,

        22,     -6389,

       -87,        44,

       -39,    180182,

     -4186,     -2227,

      -503,      -208,

       976,      -512,     -1016,     -4403,

        16,     25357,

       565,      -891,

        28,     38836,

      -659,      -855,

      -200,       619,

         1,     -1780,

        24,     17531,

      1053,       335,

       502,      2745,      -309,  -1094576,

        -1,      -263,

         3,      -394,

         0,      4152,

        -1,      -467,

      -339,   -132142,

       -14,    -11676,

        29,     63360,

         0,       129,

       -12,     -4802,

        -1,      -180,

         2,      3424,

         0,        45,

        -1,      -155,

        -1,       -74,

         0,      -134,

        -4,     -1588,

        -1,     -3424,

         2,       119,

       -57,    -27417,

         0,       168,

         0,       206,

       -21,    -80823,

         8,      2324,

        -4,       835,

       396,        -6,      -244,   -308040,

         0,      -294,

       -56,     -1332,

         2,      -110,

       306,      4060,       -58,  -1644587,

      -155,      -210,

        67,     -1642,

        27,     32070,

         9,      3592,

        -8,       -17,

        -1,         2,

        -9,     -4580,

         5,      1940,

         5,     -1217,

        62,      -862,

      -522,     -1163,

         0,         0,

     -1646,         1,      1169,       109,      1395,  23634968,

         2,         0,

      -512,      1148,

        -9,        32,

         0,     -2749,

       969,        -4,

         2,       351,

         1,       105,

       152,     97075,

       -55,    -84524,

       -49,       607,        29,   -242117,

         4,      6518,

         5,       -77,

         1,       -39,

         9,         1,

         5,      -231,

      2296,       -10,     -2453,         3,       138,   7689964,

         0,      -689,

         1,     12635,

      5444,       372,

         9,     -1843,

      3485,      -560,

       909,         3,      -742,        -1,       -66,  -6945414,

         8,        -7,

        -3,      -181,

         1,      1282,

      -115,    -76497,

         3,       -90,

         1,      -142,

        -5,     10263,

       -33,     -9479,

         0,      -657,

        19,     15040,

        -2,      1084,

        -1,      -295,

      -161,       403,

         0,      -175,

        -1,      -392,

        -4,     -3096,

        -9,     -2928,

         1,       332,

       -56,    -43733,

         1,       572,

         2,      -660,

         4,      7540,

         4,      -659,

      -505,   -382941,

        -3,       256,

         2,        30,

        76,    145296,

        -4,      4115,

         9,      6329,

        11,      6962,

         0,      -338,

         1,       153,

         2,      -147,

      -270,       211,

      -519,        97,       385,   1912268,

        -1,      1495,

        -3,       223,

       254,    152832,

        15,      6709,

        -8,     -5835,

       -12,    -29016,

         0,     -5641,

        -2,      4217,

        -2,     -1176,

      -496,         4,       331,    360972,

         1,       918,

        -2,      -136,

      -111,   -762309,

         2,      -610,

         1,        96,

       -11,     -5468,

         0,      4856,

         0,        -2,

       -16,      -668,

        20,      1963,

        -1,       -95,

         0,       -83,

         0,       -96,

        -2,      -902,

         0,     -1523,

       -14,     -5691,

       -10,    -18650,

         0,       131,

         0,       128,

         0,       243,

         1,       644,

       -76,    138046,

        -1,       316,

         8,      2703,

        15,     11758,

        -1,        16,

         2,     -2867,

         5,      6504,

       180,    143085,

        -1,       224,

        86,    -96384,

         1,       471,

         0,      -407,

        -3,     -2886,

        -1,      -919,

         0,       742,

        -2,      1145,

        59,     19306,

         0,        67,

       -61,    -67569,

       -14,     11725,

        -1,      -395,

         1,       772,

        -1,       -49,

       563,       871,      1951,       -22,

        -1,      -711,

         0,       240,

        -6,     -3912,

         1,      2812,

         0,       106,

         0,       378,

         4,     19609,

         6,      3331,

         2,       930,

         0,       -37,

         0,      -502,

         0,     -1257,

        21,     10531,

         2,    -16662,

         0,      -267,

        -1,        22,

         6,      1104,

       -14,     -5544,

         1,      -250,

        -1,      1248,

         0,       325,

         2,      2163,

         2,     -1421,

         0,        73,

        -1,       -67,

         2,       771,

        -1,     -2085,

         0,        67,

         1,       226,

         1,       212,

         0,        56,

};

static CHAR lonargs[] = {
  0,  0,
  4,  1, 10,  3, 11,-18,  2, 13,  3,  0,
  4, -2, 12,  2, 13,  5,  2, -6,  3,  0,
  4,  2, 10, -2, 13, -2,  3,  3,  5,  0,
  3,  1, 10, -1, 14,  1,  5,  0,
  4, -2, 10,  2, 13, -5,  3, -6,  4,  0,
  2,  1, 12, -1, 14,  0,
  4, -2, 10,  2, 13,  2,  3, -2,  5,  0,
  2,  1,  3, -2,  4,  0,
  4, -1, 12,  1, 13, -3,  2,  6,  3,  0,
  1,  1,  5,  0,
  4, -1, 10,  2, 13,-15,  2, 13,  3,  0,
  4,  2, 10, -2, 13, -3,  2,  3,  3,  0,
  2,  2, 12, -2, 14,  0,
  3,  1, 10, -1, 11, -1, 13,  0,
  2, 11,  2,-18,  3,  0,
  2,  3,  2, -5,  3,  0,
  2,  2,  3, -4,  4,  0,
  2,  5,  2, -8,  3,  0,
  2,  1, 10, -1, 12,  0,
  4, -1, 12,  1, 14,  3,  2, -5,  3,  0,
  3,  6, 12, -6, 13, -4,  2,  0,
  2,  3,  3, -6,  4,  0,
  2,  5,  3, -9,  4,  0,
  4, -1, 10,  2, 13,-18,  2, 18,  3,  0,
  3,  2, 10, -2,  3, -2, 13,  0,
  2,  2,  2, -3,  3,  0,
  4,  2, 10, -2, 13, -4,  2,  5,  3,  0,
  2,  4,  3, -7,  4,  0,
  2,  2, 10, -2, 12,  0,
  2,  3,  3, -5,  4,  0,
  3,  3,  2, -4,  3, -1,  4,  2,
  3,  1,  2, -5,  3,  7,  4,  2,
  4, -2, 10,  2, 13,  5,  2, -6,  3,  0,
  2,  1,  2, -2,  3,  0,
  2,  2,  3, -3,  4,  0,
  4, -1, 12,  1, 14,  4,  2, -6,  3,  0,
  2,  1,  3, -1,  4,  0,
  2,  4,  2, -6,  3,  0,
  4,  2, 10, -2, 13, -2,  2,  2,  3,  0,
  2,  1,  2, -1,  3,  0,
  3,  2, 10,  1, 11, -2, 12,  0,
  2,  1,  3, -3,  5,  0,
  3,  2, 10, -1,  3, -2, 13,  0,
  2,  4,  3, -6,  4,  0,
  2,  1,  3, -2,  5,  0,
  2,  3,  3, -4,  4,  0,
  2,  3,  2, -4,  3,  0,
  2,  1, 10, -1, 13,  0,
  2,  1,  3, -1,  5,  0,
  2,  1,  3, -2,  6,  0,
  2,  2,  3, -2,  4,  0,
  2,  1,  3, -1,  6,  0,
  1,  1, 11,  2,
  2,  1,  3,  1,  5,  0,
  3,  1, 11, -2, 12,  2, 13,  0,
  3,  1, 10, -2, 12,  1, 13,  0,
  2,  2,  2, -2,  3,  0,
  3,  2, 10, -1, 11, -2, 12,  0,
  2,  4,  3, -5,  4,  0,
  2,  3,  3, -3,  4,  0,
  3,  4, 10, -2, 12, -2, 13,  0,
  1,  1,  2,  0,
  2,  2,  3, -3,  5,  0,
  2,  2, 10, -2, 13,  2,
  2,  2,  3, -2,  5,  0,
  2,  3,  3, -3,  2,  0,
  3,  1, 10,  1, 11, -1, 13,  0,
  3,  3,  3, -1,  2,  1,  4,  0,
  2,  2,  3, -1,  5,  0,
  3,  1, 10,  1, 12, -2, 13,  0,
  1,  2, 11,  0,
  2,  2, 12, -2, 13,  0,
  2,  2,  2, -1,  3,  0,
  2,  4,  2, -4,  3,  0,
  2,  3, 10, -3, 13,  0,
  4,  2, 10, -2, 13,  1,  3, -1,  5,  0,
  2,  3,  3, -3,  5,  0,
  3,  2, 10,  1, 11, -2, 13,  0,
  3,  1, 10,  2, 12, -3, 13,  0,
  1,  3, 11,  0,
  3,  1, 11,  2, 12, -2, 13,  0,
  2,  5,  2, -5,  3,  0,
  2,  4, 10, -4, 13,  0,
  2,  6,  2, -6,  3,  0,
  3,  2, 10,  2, 11, -2, 13,  0,
  3,  2, 10,  2, 12, -4, 13,  0,
  3,  2, 11,  2, 12, -2, 13,  0,
  3,  4, 10,  1, 11, -4, 13,  0,
  4,  1, 10,  1, 11,  2, 12, -4, 13,  0,
  3,  1, 10,  3, 11, -2, 13,  0,
  3,  3, 10,  1, 11, -4, 13,  0,
  4,  1, 10, -2, 11, -2, 12,  2, 13,  0,
  3,  1, 10,  2, 12, -4, 13,  0,
  3,  1, 10,  2, 11, -2, 13,  0,
  3,  2, 10,  1, 11, -3, 13,  0,
  3,  1, 13, -2,  2,  1,  4,  0,
  2,  3, 10, -4, 13,  0,
  4,  1, 10, -1, 11, -2, 12,  2, 13,  0,
  4, -1, 10,  2, 13, -2,  2,  2,  3,  0,
  2,  1, 10, -3, 11,  0,
  2,  2, 12, -3, 13,  0,
  2,  2, 11, -1, 13,  0,
  3,  1, 10,  1, 11, -2, 13,  2,
  4, -1, 10,  2, 13, -1,  3,  1,  5,  0,
  2,  2, 10, -3, 13,  0,
  3,  3, 10, -1, 11, -4, 13,  0,
  3,  4, 11,  2, 13,-11,  2,  0,
  3,  1, 10, -2, 12,  2, 13,  0,
  3,  1, 12, -2, 13,  1,  5,  0,
  2,  1, 10, -2, 11,  0,
  2,  1, 11, -1, 13,  0,
  3,  1, 10, -2, 13,  1,  5,  0,
  3,  1, 10, -2,  3,  2,  5,  0,
  3, 18,  2,-18,  3,  2,  5,  0,
  3,  1, 10, -3,  2,  3,  3,  0,
  3,  2, 13, 16, 11,-18,  2,  0,
  2,  1, 10, -2, 13,  2,
  3,  1, 13, -2, 10,  5,  4,  0,
  3,  2, 12, -2, 14,  7,  2,  0,
  3,  1, 10, -2,  3,  3,  5,  0,
  3,  1, 12, -1,  3, -2,  5,  1,
  3,  2, 10, -1, 11, -3, 13,  0,
  3,  3, 10, -2, 12, -2, 13,  0,
  2,  8,  2, -1,  3,  0,
  4,  1, 10,  1, 11, -2, 12,  2, 13,  0,
  3,  1, 12, -2, 14,  2,  4,  0,
  2,  1, 10, -1, 11,  0,
  3,  8,  2, -5,  3,  8,  4,  0,
  2, 18,  2,-17,  3,  0,
  3,  1, 10, -1,  3,  1,  5,  0,
  1,  1, 13,  0,
  3,  1, 10, -1, 11, -2, 13,  1,
  3,  1, 10,  1, 11, -2, 12,  0,
  3,  1, 10, -1,  2,  1,  3,  0,
  3,  2, 13, -8,  2,  1,  3,  0,
  2,  3, 10, -2, 12,  0,
  3,  2, 10, -1, 12, -1,  5,  0,
  3,  3, 10, -2, 11, -2, 13,  0,
  3,  2, 10, -1, 11, -1, 13,  0,
  3,  2, 10,  1,  3, -2,  4,  0,
  3,  1, 12, -1, 10, -1, 14,  0,
  4, -1, 10,  2, 13,  2,  3, -3,  5,  0,
  3,  2, 13, -4,  2, -5,  3,  0,
  3,  2, 12,  2, 10, -3, 14,  1,
  2,  1, 12, -2,  5,  0,
  2, 10,  2, -3, 11,  0,
  1,  1, 10,  2,
  3,  1, 12, -2, 10, -2,  5,  0,
  2, 18,  2,-16,  3,  0,
  2,  2, 12, -3, 14,  1,
  3,  1, 13,-17,  2,  2, 11,  0,
  3,  1, 10,  1,  2, -3,  4,  0,
  3,  1, 12,  1, 10, -1, 14,  0,
  4, -1, 10,  2, 13,  2,  3, -2,  5,  0,
  3,  1, 10, -1,  3,  2,  4,  0,
  4, -1, 10,  2, 13,  3,  2, -3,  3,  0,
  4,  1, 10, -2, 11,  2, 12, -2, 13,  0,
  2,  1, 11,  1, 13,  0,
  3,  8,  2,  2,  3, -3,  4,  0,
  1,  1, 12,  0,
  3,  6,  2,  9,  3,-10,  4,  1,
  3,  1, 10, -2, 11, -2, 13,  0,
  2,  8,  2,  1,  4,  0,
  2,  1, 10, -2, 12,  0,
  2, 11,  2, -4,  3,  0,
  3,  3, 11, -2, 14,  6,  2,  0,
  3,  3, 10, -1, 11, -2, 13,  0,
  2,  2, 10, -1, 13,  0,
  3,  1, 12, -2,  2,  4,  3,  0,
  2,  1, 10,  1, 11,  1,
  4,  1, 10, -1, 11,  2, 12, -2, 13,  0,
  2,  2, 11,  1, 13,  0,
  2,  2, 12, -1, 13,  0,
  3,  1, 10, -1, 11, -2, 12,  0,
  2,  3, 10, -2, 13,  0,
  2,  1, 10,  2, 11,  0,
  3,  1, 10,  2, 12, -2, 13,  0,
  3,  1, 11,  2, 12, -1, 13,  0,
  3,  3, 10,  1, 11, -2, 13,  0,
  2,  1, 10,  3, 11,  0,
  4,  1, 10,  1, 11,  2, 12, -2, 13,  0,
  2,  5, 10, -4, 13,  0,
  3,  3, 10,  2, 11, -2, 13,  0,
  3,  3, 10,  2, 12, -4, 13,  0,
  2,  4, 11, -2, 13,  0,
  3,  2, 10,  2, 11, -4, 13,  0,
  2,  3, 11, -2, 13,  0,
  3,  1, 10,  2, 11, -3, 13,  0,
  3,  2, 10,  1, 11, -4, 13,  0,
  2,  3, 10, -5, 13,  0,
  2,  2, 12, -4, 13,  0,
  2,  2, 11, -2, 13,  0,
  3,  1, 10,  1, 11, -3, 13,  0,
  3,  2, 13, -2,  3,  2,  5,  0,
  2,  2, 10, -4, 13,  1,
  4,  2, 10, -1, 11, -2, 12,  2, 13,  0,
  2,  2, 14, -2,  2,  0,
  3,  1, 10, -2, 12,  3, 13,  0,
  2,  1, 11, -2, 13,  1,
  3,  2, 14, -2,  2,  3,  5,  0,
  3,  2, 13, -1,  3,  1,  5,  0,
  2,  1, 10, -3, 13,  0,
  3,  2, 10, -1, 11, -4, 13,  0,
  4,  2, 10,  1, 11, -2, 12, -2, 13,  0,
  3,  2, 13,  1, 14, -8,  2,  0,
  3,  2, 10, -2, 12,  2, 13,  0,
  2,  2, 10, -2, 11,  0,
  3,  1, 10, -1, 11,  1, 13,  0,
  3,  1, 12, -2, 13, -1, 14,  0,
  3,  2, 13, -8,  2, 13,  3,  0,
  3,  3, 10, -5, 13,  5,  4,  0,
  1,  2, 13,  2,
  3,  3, 10, -1, 13,  5,  4,  0,
  3,  2, 13,  8,  2,-13,  3,  0,
  2,  2, 11, -2, 12,  0,
  3,  1, 10, -1, 11, -3, 13,  0,
  3,  1, 10, -1, 12, -2, 13,  0,
  3,  2, 10, -2, 11, -4, 13,  0,
  3,  2, 10, -2, 12, -2, 13,  0,
  2,  2, 10, -1, 11,  0,
  2,  1, 10,  1, 13,  0,
  2,  1, 11,  2, 13,  1,
  2,  1, 11, -2, 12,  0,
  3,  1, 10, -2, 12, -1, 13,  0,
  2,  4, 10, -2, 12,  0,
  2,  3, 10, -1, 12,  0,
  3,  3, 10, -1, 11, -1, 13,  0,
  1,  2, 10,  2,
  2,  2, 14, -2,  5,  0,
  3,  1, 10,  1, 11,  1, 13,  0,
  2,  1, 10,  1, 12,  0,
  2,  2, 11,  2, 13,  0,
  2,  1, 12,  1, 14,  0,
  1,  2, 12,  2,
  2,  1, 10, -3, 12,  0,
  3,  4, 10, -1, 11, -2, 13,  0,
  2,  3, 10, -1, 13,  0,
  2,  2, 10,  1, 11,  0,
  3,  1, 10,  2, 11,  1, 13,  0,
  3,  1, 10,  2, 12, -1, 13,  0,
  2,  1, 11,  2, 12,  0,
  2,  4, 10, -2, 13,  0,
  2,  2, 10,  2, 11,  0,
  3,  2, 10,  2, 12, -2, 13,  0,
  2,  4, 12, -2, 13,  0,
  3,  4, 10,  1, 11, -2, 13,  0,
  3,  2, 13, 11,  2,-13,  3,  0,
  3,  1, 10,  3, 11, -4, 13,  0,
  3,  3, 10,  1, 11, -6, 13,  0,
  3,  1, 10,  2, 11, -4, 13,  0,
  2,  3, 10, -6, 13,  0,
  3,  1, 10, -3, 11,  2, 13,  0,
  3,  1, 10,  1, 11, -4, 13,  0,
  2,  2, 10, -5, 13,  0,
  3,  1, 10, -2, 12,  4, 13,  0,
  3,  1, 10, -2, 11,  2, 13,  0,
  2,  1, 11, -3, 13,  0,
  2,  1, 10, -4, 13,  0,
  4,  1, 10,  2, 11, -2, 12, -2, 13,  0,
  3,  3, 10, -2, 12, -4, 13,  0,
  3,  1, 10, -1, 11,  2, 13,  0,
  1,  3, 13,  0,
  3,  1, 10, -1, 11, -4, 13,  0,
  4,  1, 10,  1, 11, -2, 12, -2, 13,  0,
  3,  3, 10, -2, 12,  2, 13,  0,
  2,  3, 10, -2, 11,  0,
  3,  2, 10, -1, 11,  1, 13,  0,
  3,  1, 12,  2, 13, -2,  5,  0,
  2,  1, 10,  2, 13,  1,
  2,  1, 11,  3, 13,  0,
  3,  1, 10, -2, 11, -4, 13,  0,
  3,  1, 10, -2, 12, -2, 13,  0,
  2,  3, 10, -1, 11,  0,
  2,  2, 10,  1, 13,  0,
  3,  1, 10,  1, 11,  2, 13,  0,
  3,  1, 10, -1, 11,  2, 12,  0,
  2,  2, 12,  1, 13,  0,
  4,  1, 10, -1, 11, -2, 12, -2, 13,  0,
  1,  3, 10,  1,
  3,  2, 10,  1, 11,  1, 13,  0,
  3,  1, 10,  2, 11,  2, 13,  0,
  2,  1, 10,  2, 12,  0,
  3,  1, 11,  2, 12,  1, 13,  0,
  2,  4, 10, -1, 13,  0,
  2,  3, 10,  1, 11,  0,
  3,  1, 10,  1, 11,  2, 12,  0,
  4,  1, 10, -1, 11,  4, 12, -2, 13,  0,
  2,  5, 10, -2, 13,  0,
  3,  3, 10,  2, 12, -2, 13,  0,
  3,  1, 10,  4, 12, -2, 13,  0,
  3,  2, 10,  2, 11, -6, 13,  0,
  2,  3, 11, -4, 13,  0,
  3,  2, 10,  1, 11, -6, 13,  0,
  2,  2, 11, -4, 13,  0,
  2,  2, 10, -6, 13,  0,
  2,  1, 11, -4, 13,  0,
  2,  1, 10, -5, 13,  0,
  3,  2, 10, -1, 11, -6, 13,  0,
  4,  2, 10,  1, 11, -2, 12, -4, 13,  0,
  3,  2, 10, -2, 11,  2, 13,  0,
  1,  4, 13,  0,
  3,  2, 11, -2, 12, -2, 13,  0,
  3,  2, 10, -2, 12, -4, 13,  0,
  3,  2, 10, -1, 11,  2, 13,  0,
  2,  1, 10,  3, 13,  0,
  2,  1, 11,  4, 13,  0,
  3,  1, 11, -2, 12, -2, 13,  0,
  2,  2, 10,  2, 13,  0,
  3,  1, 10,  1, 11,  3, 13,  0,
  2,  2, 12,  2, 13,  0,
  2,  4, 10, -1, 11,  0,
  2,  3, 10,  1, 13,  0,
  3,  2, 10,  1, 11,  2, 13,  0,
  3,  2, 10, -1, 11,  2, 12,  0,
  3,  1, 10,  2, 12,  1, 13,  0,
  3,  1, 11,  2, 12,  2, 13,  0,
  1,  4, 10,  0,
  3,  3, 10,  1, 11,  1, 13,  0,
  2,  2, 10,  2, 12,  0,
  1,  4, 12,  0,
  2,  4, 10,  1, 11,  0,
  3,  2, 10,  1, 11,  2, 12,  0,
  2,  6, 10, -2, 13,  0,
  3,  5, 12, -1, 14,  2,  4,  1,
  3,  1, 10,  1, 11, -6, 13,  0,
  3,  1, 10, -2, 11,  4, 13,  0,
  2,  1, 10, -6, 13,  0,
  3,  1, 10, -1, 11,  4, 13,  0,
  3,  1, 10, -1, 11, -6, 13,  0,
  4,  1, 10,  1, 11, -2, 12, -4, 13,  0,
  2,  1, 10,  4, 13,  0,
  3,  1, 10, -2, 12, -4, 13,  0,
  3,  3, 10, -1, 11,  2, 13,  0,
  2,  2, 10,  3, 13,  0,
  3,  1, 10,  1, 11,  4, 13,  0,
  4,  1, 10, -1, 11,  2, 12,  2, 13,  0,
  2,  3, 10,  2, 13,  0,
  3,  1, 10,  2, 12,  2, 13,  0,
  3,  3, 10,  1, 11,  2, 13,  0,
  3,  1, 10, -1, 11,  4, 12,  0,
  1,  5, 10,  0,
  2,  3, 10,  2, 12,  0,
  2,  1, 11, -6, 13,  0,
  1,  6, 13,  0,
  3,  2, 10, -1, 11,  4, 13,  0,
  2,  2, 10,  4, 13,  0,
  2,  2, 12,  4, 13,  0,
  3,  4, 10, -1, 11,  2, 13,  0,
  3,  2, 10,  1, 11,  4, 13,  0,
  2,  4, 10,  2, 13,  0,
  3,  2, 10,  2, 12,  2, 13,  0,
  1,  6, 10,  0,
  2,  1, 10,  6, 13,  0,
  2,  3, 10,  4, 13,  0,
  2,  5, 10,  2, 13,  0,
 -1
};

/* Total terms = 356, small = 356 */
static struct plantbl liblon = {
  /*  {  0, 18, 18, 10,  3,  2,  0,  0,  0,  6, 16,  6,  6,  3,}, */
  /* Use max of liblon, liblat.  */
  {  0, 18, 18, 20,  4,  2,  0,  0,  0,  9, 16,  7,  9,  9,},
 2,
 lonargs,
 lontabl,
 lontabb,
 lontabr,
 3.850000e+05,
 36525.0,
 1.0e-4,
};


/* given a Julian date, return the lunar libration in lat and long, in rads.
 */
void
llibration (double JD, double *llatp, double *llonp)
{
	double lg, lt;	/* arc seconds */
	
	lg = gplan (JD, &liblon);
	lt = gplan (JD, &liblat);

	*llonp = degrad (lg/3600.0);
	*llatp = degrad (lt/3600.0);
}

